﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Threading;

namespace ConsoleGames.Games2024
{
    public class Tetris : Game
    {

        public override string Name => "Tetris";
        public override string Description => "\nDies ist eine Version des klassischen Spiels namens \"Tetris\". \nBei Tetris besteht Ihre Aufgabe darin, Linien zu vervollständigen, indem Sie unterschiedlich geformte Steine (Tetrominos) bewegen, die langsam auf das Spielfeld fallen. \nDie vervollständigten Linien verschwinden und bringen Ihnen Punkte ein, und Sie können fortfahren, die frei gewordenen Felder zu füllen. \nDas Spiel endet, wenn die nicht abgeräumten Linien den oberen Rand des Spielfelds erreichen.";
        public override string Rules => "\n\nPunktzahlanforderungen:\n\nUm zum 2. Level zu gelangen, müssen Sie 200 oder mehr Punkte erreichen. Für das dritte Level müssen Sie 500 Punkte oder mehr erzielen und schliesslich für das dritte Level mindestens 700 Punkte erreichen.\n\nSchwierigkeitsgrade:\n\nJedes Level unterscheidet sich nicht nur in der Anzahl der Punkte, die für den Übergang zum nächsten Level erforderlich sind, sondern auch in der Geschwindigkeit der fallenden Tetrominos.\n\n";
        public override string Credits => "Zhabrovets Andrii, anzhabro@ksr.ch";
        public override int Year => 2024;
        public override bool TheHigherTheBetter => true;
        public override int LevelMax => 3;
        public override Score HighScore { get; set; }


        private class Tetromino
        {
            // Arrays to define types and colors of Tetrominos
            private char[] Tetrominostypes = { 'I', 'J', 'L', 'O', 'S', 'T', 'Z' };
            private ConsoleColor[] Colors = { ConsoleColor.Cyan, ConsoleColor.Blue, ConsoleColor.DarkYellow, ConsoleColor.Yellow, ConsoleColor.Green, ConsoleColor.Magenta, ConsoleColor.Red };

            // Tetromino properties
            public int x;
            public int y;
            public char type;
            public int width;
            public int height;
            public int[][] body;
            public int[][] bottom;
            public ConsoleColor Color;

            // Tetromino presets for different rotations
            private int[][][][] Presets = {
            // I
            new int[][][]{

                new int[][]{
                    new int[]{0,1},
                    new int[]{1,1},
                    new int[]{2,1},
                    new int[]{3,1}
                },
                new int[][]{
                    new int[]{2,0},
                    new int[]{2,1},
                    new int[]{2,2},
                    new int[]{2,3}
                },
                new int[][]{
                    new int[]{0,2},
                    new int[]{1,2},
                    new int[]{2,2},
                    new int[]{3,2}
                },
                new int[][]{
                    new int[]{1,0},
                    new int[]{1,1},
                    new int[]{1,2},
                    new int[]{1,3}
                }

            },

            // J
            new int[][][]{
                new int[][]{
                    new int[]{0,0},
                    new int[]{0,1},
                    new int[]{1,1},
                    new int[]{2,1}
                },
                new int[][]{
                    new int[]{1,0},
                    new int[]{1,1},
                    new int[]{1,2},
                    new int[]{2,0}
                },
                new int[][]{
                    new int[]{0,1},
                    new int[]{1,1},
                    new int[]{2,1},
                    new int[]{2,2}
                },
                new int[][]{
                    new int[]{0,2},
                    new int[]{1,0},
                    new int[]{1,1},
                    new int[]{1,2}
                }



            },

            // L
            new int[][][]{
                new int[][]{
                    new int[]{0,1},
                    new int[]{1,1},
                    new int[]{2,0},
                    new int[]{2,1}
                },
                new int[][]{
                    new int[]{1,0},
                    new int[]{1,1},
                    new int[]{1,2},
                    new int[]{2,2}
                },
                new int[][]{
                    new int[]{0,1},
                    new int[]{0,2},
                    new int[]{1,1},
                    new int[]{2,1}
                },
                new int[][]{
                    new int[]{0,0},
                    new int[]{1,0},
                    new int[]{1,1},
                    new int[]{1,2}
                }

            },

            // O
            new int[][][]{
                new int[][]{
                    new int[]{1,0},
                    new int[]{1,1},
                    new int[]{2,0},
                    new int[]{2,1}
                }

            },

            // S
            new int[][][]{
                new int[][]{
                    new int[]{0,1},
                    new int[]{1,0},
                    new int[]{1,1},
                    new int[]{2,0}
                },
                new int[][]{
                    new int[]{1,0},
                    new int[]{1,1},
                    new int[]{2,1},
                    new int[]{2,2}
                },
                new int[][]{
                    new int[]{0,2},
                    new int[]{1,1},
                    new int[]{1,2},
                    new int[]{2,1}
                },
                new int[][]{
                    new int[]{0,0},
                    new int[]{0,1},
                    new int[]{1,1},
                    new int[]{1,2}
                }

            },


            // T
            new int[][][]{
                new int[][]{
                    new int[]{0,1},
                    new int[]{1,0},
                    new int[]{1,1},
                    new int[]{2,1}
                },
                new int[][]{
                    new int[]{1,0},
                    new int[]{1,1},
                    new int[]{1,2},
                    new int[]{2,1}
                },
                new int[][]{
                    new int[]{0,1},
                    new int[]{1,1},
                    new int[]{1,2},
                    new int[]{2,1}
                },
                new int[][]{
                    new int[]{0,1},
                    new int[]{1,0},
                    new int[]{1,1},
                    new int[]{1,2}
                }


            },


            // Z
            new int[][][]{
                new int[][]{
                    new int[]{0,0},
                    new int[]{1,0},
                    new int[]{1,1},
                    new int[]{2,1}
                },
                new int[][]{
                    new int[]{1,1},
                    new int[]{1,2},
                    new int[]{2,0},
                    new int[]{2,1}
                },
                new int[][]{
                    new int[]{0,1},
                    new int[]{1,1},
                    new int[]{1,2},
                    new int[]{2,2}
                },
                new int[][]{
                    new int[]{0,1},
                    new int[]{0,2},
                    new int[]{1,0},
                    new int[]{1,1}
                }
            }

        };
            // Random instance for selecting Tetromino type
            Random rand = new Random();


            public Tetromino(int _x, int _y)
            {

                x = _x;
                y = _y;
                type = Tetrominostypes[rand.Next(0, Tetrominostypes.Length)];
                body = CalculateBody();
                width = CalculateWidth();
                height = CalculateHeight();
                bottom = CalculateBottom();
                Color = SetColor();


            }

            // Set color based on Tetromino type
            private ConsoleColor SetColor()
            {
                int typeIndex = Array.IndexOf(Tetrominostypes, type);
                return Colors[typeIndex];
            }

            public void SetNext(int[,] GameBorder, List<Tetromino> tetrominos)
            {
                int TypeIndex = Array.IndexOf(Tetrominostypes, type);
                int[][][] Rotations = Presets[TypeIndex];
                int currentIndex = Array.IndexOf(Rotations, body);
                int nextIndex;
                if (currentIndex < Rotations.Length - 1)
                {
                    nextIndex = currentIndex + 1;
                }
                else
                {
                    nextIndex = 0;
                }
                if (!CheckRotationCollision(Rotations[nextIndex], x, y, GameBorder, tetrominos))
                {

                    this.body = Rotations[nextIndex];
                    width = CalculateWidth();
                    height = CalculateHeight();
                    bottom = CalculateBottom();
                }

            }

            // Check if the rotation causes a collision
            private bool CheckRotationCollision(int[][] ActiveTetromino, int newX, int newY, int[,] GameBorder, List<Tetromino> tetrominos)
            {


                foreach (int[] block in ActiveTetromino)
                {

                    if (block[0] + newX <= GameBorder[0, 0] || block[0] + newX >= GameBorder[0, 1] || block[1] + newY <= GameBorder[1, 0] || block[1] + newY >= GameBorder[1, 1])
                    {

                        return true;
                    }

                    foreach (Tetromino fallenpiece in tetrominos)
                    {

                        foreach (int[] fallenblock in fallenpiece.body)
                        {

                            if (block[0] + newX == fallenblock[0] + fallenpiece.x && block[1] + newY == fallenblock[1] + fallenpiece.y)
                            {

                                return true;
                            }
                        }
                    }
                }
                return false;
            }

            // Move the Tetromino to the lowest position possible
            // This Quickfall function has to be completely rewritten. Setting a precise location is wrong thus it
            // Because of the fact that "I" tetromino in horizontal position has height of 1 QuickFall function subtracts 1 from the HighestY, but at the same time I has its local coordinates Y that equal 2 or 1.
            // Turns out quick fall breaks with other types of blocks as well. Has to be discussed.
            public void QuickFall(int BorderX, int BorderY, List<Tetromino> tetrominos)
            {

                List<int> HighPointsAtX = new List<int>();
                int[] HighestPoints = new int[width];
                for (int i = 0; i < width; i++)
                {
                    foreach (Tetromino t in tetrominos)
                    {
                        foreach (int[] block in t.body)
                        {
                            if (x + bottom[i][0] == block[0] + t.x && y + bottom[i][1] < block[1] + t.y)
                            {
                                // Console.WriteLine(CalculatDistanceToCollision(GameBorder[1,1], tetrominos, CurrentTetromino)[i]);
                                HighPointsAtX.Add(block[1] + t.y);
                            }
                            else
                            {

                                HighPointsAtX.Add(BorderY);
                            }
                        }
                    }
                    int LN = BorderY;
                    foreach (int n in HighPointsAtX)
                    {

                        if (n < LN)
                        {
                            LN = n;
                        }
                    }

                    HighestPoints[i] = (LN - 1);
                    HighPointsAtX.Clear();
                }

                int LNum = HighestPoints[0];
                int BottomPoint = bottom[0][1];
                for (int i = 0; i < HighestPoints.Count(); i++)
                {

                    if (HighestPoints[i] < LNum)
                    {
                        LNum = HighestPoints[i];
                        BottomPoint = bottom[i][1];
                    }
                }
                y += LNum - y - BottomPoint;


            }

            // Calculate the height of the Tetromino
            private int CalculateHeight()
            {
                int HighPoint = 0;
                int LowPoint = body[0][1];
                foreach (int[] block in body)
                {
                    if (block[1] > HighPoint)
                    {
                        HighPoint = block[1];
                    }
                    else if (block[1] < LowPoint)
                    {
                        LowPoint = block[1];
                    }
                }
                return HighPoint - LowPoint + 1;
            }

            // Calculate the width of the Tetromino
            private int CalculateWidth()
            {
                int HighPoint = 0;
                int LowPoint = body[0][0];

                foreach (int[] block in body)
                {
                    if (block[0] > HighPoint)
                    {
                        HighPoint = block[0];
                    }
                    else if (block[0] < LowPoint)
                    {
                        LowPoint = block[0];
                    }
                }
                return HighPoint - LowPoint + 1;
            }

            // Calculate the initial body of the Tetromino
            private int[][] CalculateBody()
            {
                int typeIndex = Array.IndexOf(Tetrominostypes, type);
                int rotationIndex = rand.Next(Presets[typeIndex].Length);
                int[][] preset = Presets[typeIndex][rotationIndex];

                return preset;

            }

            // Calculate the bottom points of the Tetromino
            private int[][] CalculateBottom()
            {
                List<int> XPoints = new List<int>();

                foreach (int[] block in body)
                {

                    if (!XPoints.Contains(block[0]))
                    {

                        XPoints.Add(block[0]);
                    }
                }

                int[][] BottomPoints = new int[XPoints.Count][];

                int Bottom = 0;
                for (int l = 0; l < XPoints.Count; l++)
                {
                    foreach (int[] block in body)
                    {
                        if (block[0] == XPoints[l] && block[1] > Bottom)
                        {
                            Bottom = block[1];
                        }
                    }

                    BottomPoints[l] = new int[] { XPoints[l], Bottom };
                    Bottom = 0;
                }
                return BottomPoints;


            }
        }

        public override Score Play(int level = 1)
        {

            Score score = new Score();
            score.LevelCompleted = false;


            bool levelCompleted = false;

            if (level > LevelMax) level = LevelMax;

            List<Tetromino> tetrominos = new List<Tetromino> { };
            Console.CursorVisible = false;

            int FieldWidth = 10;
            int FieldHeight = 16;

            int[] StartingPoint = { 10, 10 };


            int Points = 0;
            int[] ScoreRequirement = { 200, 500, 700 };
            int[] SpeedLevel = { 700, 500, 300 };




            int[] HighestPoints = new int[FieldWidth];
            int[,] GameBorder = { { StartingPoint[0], StartingPoint[0] + FieldWidth + 1 }, { StartingPoint[0], StartingPoint[0] + FieldHeight + 1 } };
            int XCoordinate = (GameBorder[0, 0] + (FieldWidth / 2));
            int YCoordinate = GameBorder[1, 0];


            Tetromino CurrentTetromino = new Tetromino(XCoordinate, YCoordinate);
            Tetromino NextTetromino = new Tetromino(XCoordinate, YCoordinate);

            CurrentTetromino.y = YCoordinate - CurrentTetromino.height;
            NextTetromino.y = YCoordinate - NextTetromino.height;

            Stopwatch stopwatch = new Stopwatch();
            bool NextBlock = false;

            WelcomeScreen();

            Console.Clear();
            ClearGameField(FieldHeight, GameBorder, FieldWidth);

            while (true)
            {

                stopwatch.Start();

                NextBlock = false;

                while (!NextBlock)
                {


                    FindHighestBlocks(tetrominos, GameBorder, ref HighestPoints);


                    CheckMovement(ref CurrentTetromino, GameBorder, HighestPoints, tetrominos);


                    DrawGameUI(FieldHeight, FieldWidth, GameBorder, CurrentTetromino, tetrominos, StartingPoint, NextTetromino, Points, level);

                    DestroyFullRows(GameBorder[1, 1], HighestPoints, GameBorder[0, 0], tetrominos, ref Points, FieldWidth);

                    UpdateGameState(ref stopwatch, SpeedLevel, level, ref CurrentTetromino, GameBorder, ref tetrominos, ref NextTetromino, XCoordinate, YCoordinate, ref NextBlock);

                    Thread.Sleep(60);

                    FindHighestBlocks(tetrominos, GameBorder, ref HighestPoints);
                }



                if (CheckTopCollision(HighestPoints, GameBorder[1, 0]))
                {
                    CheckGameState(Points, ScoreRequirement, ref levelCompleted, level);
                    GameOverScreen(Points, level, levelCompleted, LevelMax);
                    score.Level = level;
                    score.LevelCompleted = levelCompleted;
                    score.Points = Points;
                    return score;
                }
            }




        }

        // Function to draw the game UI including the playing field, current Tetromino, fallen Tetrominos, next Tetromino, and game information
        static void DrawGameUI(int FieldHeight, int FieldWidth, int[,] GameBorder, Tetromino CurrentTetromino, List<Tetromino> tetrominos, int[] StartingPoint, Tetromino NextTetromino, int Points, int level)
        {

            PrintBorders(FieldHeight, GameBorder, FieldWidth);


            ClearGameField(FieldHeight, GameBorder, FieldWidth);
            PrintTetromino(CurrentTetromino, CurrentTetromino.x, CurrentTetromino.y, GameBorder[1, 0], true);


            //ClearTetromino(OldTetromino, OldTetromino.x, OldTetromino.y, GameBorder[1, 0], true);



            foreach (Tetromino block in tetrominos)
            {
                PrintTetromino(block, block.x, block.y, GameBorder[1, 0], true);
            }

            PrintSideBar(StartingPoint, FieldWidth, NextTetromino, FieldHeight, Points, GameBorder[1, 0], level);




        }


        static void ClearTetromino(Tetromino t, int x, int y, int GameBorderY, bool OnField)
        {

            Console.ResetColor();
            Console.ForegroundColor = ConsoleColor.DarkYellow;
            Console.SetCursorPosition(x, y);
            for (int i = 0; i < t.body.Length; i++)
            {
                if (OnField)
                {
                    if (t.body[i][1] + t.y > GameBorderY)
                    {
                        //  Console.BackgroundColor = t.Color;
                        Console.SetCursorPosition(x + t.body[i][0], y + t.body[i][1]);
                        Console.WriteLine("·");
                    }
                }
                else
                {
                    Console.SetCursorPosition(x + t.body[i][0], y + t.body[i][1]);
                    Console.WriteLine("·");
                }

            }

            Console.ResetColor();
        }





        // Function to display the game over screen with relevant information
        static void GameOverScreen(int Score, int Level, bool levelCompleted, int LevelMax)
        {

            // Console.Clear();
            Console.SetCursorPosition(0, 0);


            Console.WriteLine("Game Over!\n");

            if (levelCompleted)
            {
                if (Level < LevelMax)
                {
                    Console.WriteLine("You won! You have scored {0} points. You can move to level {1}.\n", Score, Level + 1);
                }
                else
                {
                    Console.WriteLine("You won! Congratulations! You have reached the maximum level.");
                }
            }
            else
            {
                if (Level < LevelMax)
                {
                    Console.WriteLine("You lost! You have scored {0} points. Unfortunately, that is not enough to move to level {1}.", Score, Level + 1);
                }
                else
                {
                    Console.WriteLine("You lost! You have scored {0} points. Unfortunately, that is not enough to win the final level.", Score);
                }

            }

            Console.ReadKey();

            Console.CursorVisible = true;


        }

        // Function to display the welcome screen and provide instructions to the player
        static void WelcomeScreen()
        {

            Console.WriteLine("▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄▄\n█▄▄░▄▄█░▄▄█▄░▄█░▄▄▀██▄██░▄▄\n███░███░▄▄██░██░▀▀▄██░▄█▄▄▀\n███░███▄▄▄██▄██▄█▄▄█▄▄▄█▄▄▄\n▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀▀");

            Console.WriteLine("Welcome to the game called \"Tetris\"\n\nIf you don\'t know the rules, read Description and Rules!\n\nControls:\n← – Move Tetromino left\n→ – Move Tetromino right\n↓ – Drop Tetromino down\n↑ - Move Tetromino down\nSpacebar - rotate Tetromino\n\nLooks like you know everything now. Click any button to start.");

            Console.ReadKey();
        }

        // Function to print the sidebar containing picture of the next Tetromino, score, and level information
        static void PrintSideBar(int[] StartingPoint, int FieldWidth, Tetromino NextTetromino, int FieldHeight, int Score, int GameBorderY, int level)
        {

            int[] StartText = { StartingPoint[0] + FieldWidth + 10, StartingPoint[1] + 1 };

            Console.SetCursorPosition(StartText[0] + 2, StartText[1]);

            Console.ForegroundColor = ConsoleColor.Yellow;

            Console.WriteLine("Next");

            for (int i = StartText[1] + 1; i < StartText[1] + 7; i++)
            {

                Console.SetCursorPosition(StartText[0] + 1, i);
                Console.WriteLine("       ");

            }

            PrintTetromino(NextTetromino, StartText[0] + 2, StartText[1] + 2, GameBorderY, false);


            Console.ForegroundColor = ConsoleColor.Yellow;

            Console.SetCursorPosition(StartText[0] + 1, StartText[1] + 7);

            Console.WriteLine("Score");

            Console.SetCursorPosition(StartText[0] + 1, StartText[1] + 9);

            Console.WriteLine(Score);

            Console.SetCursorPosition(StartText[0] + 1, StartText[1] + 12);

            Console.WriteLine("Level");

            Console.SetCursorPosition(StartText[0] + 1, StartText[1] + 14);

            Console.WriteLine(level);

            Console.ResetColor();

            for (int k = 0; k <= FieldHeight - 1; k++)
            {
                Console.SetCursorPosition(StartText[0] - 1, StartText[1] + k);
                Console.WriteLine("|");
                Console.SetCursorPosition(StartText[0] + 8, StartText[1] + k);
                Console.WriteLine("|");
            }

            for (int j = 0; j <= 9; j++)
            {

                Console.SetCursorPosition(StartText[0] - 1 + j, StartText[1] - 1);
                Console.WriteLine("_");

                Console.SetCursorPosition(StartText[0] - 1 + j, StartText[1] + FieldHeight);
                Console.WriteLine("—");

            }





        }

        static void ClearGameField(int FieldHeight, int[,] GameBorder, int FieldWidth)
        {

            string Filling = "";
            Console.ForegroundColor = ConsoleColor.DarkYellow;
            for (int k = 0; k < FieldWidth; k++)
            {
                Filling = Filling + "·";
            }
            for (int i = 0; i < FieldHeight; i++)
            {
                Console.SetCursorPosition(GameBorder[0, 0] + 1, GameBorder[1, 0] + 1 + i);
                Console.WriteLine(Filling);
            }
            Console.ResetColor();
        }
        // Function to print the borders of the playing field
        static void PrintBorders(int FieldHeight, int[,] GameBorder, int FieldWidth)
        {
            //   Console.Clear();
            Console.BackgroundColor = ConsoleColor.DarkGray;
            Console.ForegroundColor = ConsoleColor.DarkGray;

            for (int k = 0; k <= FieldHeight; k++)
            {
                Console.SetCursorPosition(GameBorder[0, 0], GameBorder[1, 0] + k);
                Console.WriteLine("|");
                Console.SetCursorPosition(GameBorder[0, 1], GameBorder[1, 0] + k);
                Console.WriteLine("|");
            }

            for (int j = 0; j <= FieldWidth + 1; j++)
            {

                Console.SetCursorPosition(GameBorder[0, 0] + j, GameBorder[1, 0]);
                Console.WriteLine("x");

                Console.SetCursorPosition(GameBorder[0, 0] + j, GameBorder[1, 1]);
                Console.WriteLine("x");

            }

            Console.ResetColor();

        }

        // Function to print the Tetromino on the console (can be used for both active and the fallen ones)
        static void PrintTetromino(Tetromino t, int x, int y, int GameBorderY, bool OnField)
        {

            Console.ForegroundColor = t.Color;
            Console.BackgroundColor = t.Color;
            Console.SetCursorPosition(x, y);
            for (int i = 0; i < t.body.Length; i++)
            {
                if (OnField)
                {
                    if (t.body[i][1] + t.y > GameBorderY)
                    {
                        //  Console.BackgroundColor = t.Color;
                        Console.SetCursorPosition(x + t.body[i][0], y + t.body[i][1]);
                        Console.WriteLine("x");
                    }
                }
                else
                {
                    Console.SetCursorPosition(x + t.body[i][0], y + t.body[i][1]);
                    Console.WriteLine("x");
                }

            }

            Console.ResetColor();
        }

        // Model

        // Function to update the game state based on elapsed time
        static void UpdateGameState(ref Stopwatch stopwatch, int[] SpeedLevel, int level, ref Tetromino CurrentTetromino, int[,] GameBorder, ref List<Tetromino> tetrominos, ref Tetromino NextTetromino, int XCoordinate, int YCoordinate, ref bool NextBlock)
        {
            if (stopwatch.ElapsedMilliseconds >= SpeedLevel[level - 1])
            {
                if (!CheckBottomCollision(CurrentTetromino, GameBorder, tetrominos))
                {


                    CurrentTetromino.y++;

                }
                else
                {
                    foreach (int[] points in CurrentTetromino.body)
                    {
                        foreach (Tetromino tetromino in tetrominos)
                        {
                            foreach (int[] block in tetromino.body)
                            {

                                if (points[1] + CurrentTetromino.y == block[1] + tetromino.y && points[0] + CurrentTetromino.x == block[0] + tetromino.x)
                                {
                                    CurrentTetromino.y--;
                                }
                                else if (points[1] + CurrentTetromino.y >= GameBorder[1, 1])
                                {

                                    CurrentTetromino.y--;
                                }
                            }
                        }
                    }

                    tetrominos.Add(CurrentTetromino);

                    foreach (Tetromino tetromino in tetrominos)
                    {
                        foreach (int[] block in tetromino.body)
                        {

                            if (block[1] + tetromino.y >= GameBorder[1, 1])
                            {

                                tetromino.y--;
                            }
                        }
                    }


                    CurrentTetromino = NextTetromino;

                    NextTetromino = new Tetromino(XCoordinate, YCoordinate);

                    NextTetromino.y = YCoordinate - NextTetromino.height;


                    NextBlock = true;

                }


                stopwatch.Restart();
            }

        }

        // Function to check if the current game state meets level completion requirements
        static void CheckGameState(int Points, int[] ScoreRequirement, ref bool levelCompleted, int level)
        {
            if (Points >= ScoreRequirement[level - 1])
            {

                levelCompleted = true;

            }
            else
            {
                levelCompleted = false;
            }
        }

        // Function to destroy full rows, shift blocks down, and update the player's score
        static void DestroyFullRows(int BorderY, int[] HighestPoints, int BorderX, List<Tetromino> tetrominos, ref int Points, int FieldWidth)
        {

            int HighestPoint = HighestPoints.Min();

            int Rows = BorderY - HighestPoint - 1;

            for (int i = 0; i <= Rows; i++)
            {

                if (IsRowFull(HighestPoint + i, BorderX, tetrominos, FieldWidth))
                {

                    ClearRow(HighestPoint + i, tetrominos);
                    ShiftDown(HighestPoint + i, tetrominos);
                    Points = Points + 100;
                }

            }

        }

        // Function to shift blocks down after clearing a row
        static void ShiftDown(int DeletedRow, List<Tetromino> tetrominos)
        {

            foreach (Tetromino t in tetrominos)
            {
                for (int i = 0; i < t.body.Length; i++)
                {

                    if (t.y + t.body[i][1] < DeletedRow)
                    {

                        t.body[i] = new int[] { t.body[i][0], t.body[i][1] + 1 };
                    }

                }
            }

        }

        // Function to clear a full row
        static void ClearRow(int row, List<Tetromino> tetrominos)
        {
            //foreach (Tetromino t in tetrominos)
            //{
            //    t.body = t.body.Where(block => t.y + block[1] != row).ToArray();
            //}

            // Same result without using array.Where()
            foreach (Tetromino t in tetrominos)
            {
                List<int[]> newBody = new List<int[]>();
                foreach (int[] block in t.body)
                {
                    if (t.y + block[1] != row)
                    {
                        newBody.Add(block);
                    }
                }

                t.body = newBody.ToArray();
            }
        }

        // Function to check if a row is full
        static bool IsRowFull(int y, int BorderX, List<Tetromino> tetrominos, int FieldWidth)
        {
            bool[] RowFull = new bool[FieldWidth];

            for (int i = 0; i < FieldWidth; i++)
            {

                RowFull[i] = false;
            }


            foreach (Tetromino piece in tetrominos)
            {
                foreach (int[] block in piece.body)
                {
                    if (piece.y + block[1] == y)
                    {

                        RowFull[(piece.x + block[0]) - BorderX - 1] = true;

                    }

                }
            }


            if (!RowFull.Contains(false))
            {

                return true;
            }

            return false;

        }

        // Function to check for user input and update Tetromino position accordingly
        static void CheckMovement(ref Tetromino CurrentTetromino, int[,] GameBorder, int[] HighestPoints, List<Tetromino> tetrominos)
        {
            if (Console.KeyAvailable)
            {
                ConsoleKeyInfo key = Console.ReadKey();


                if (key.Key == ConsoleKey.LeftArrow)
                {
                    if (!CheckSideCollision(CurrentTetromino, GameBorder, "left", tetrominos))
                    {
                        CurrentTetromino.x = CurrentTetromino.x - 1;
                    }
                }
                else if (key.Key == ConsoleKey.RightArrow)
                {
                    if (!CheckSideCollision(CurrentTetromino, GameBorder, "right", tetrominos))
                    {
                        CurrentTetromino.x = CurrentTetromino.x + 1;
                    }
                }

                else if (key.Key == ConsoleKey.Spacebar)
                {
                    CurrentTetromino.SetNext(GameBorder, tetrominos);
                }

                else if (key.Key == ConsoleKey.DownArrow)
                {
                    if (!CheckBottomCollision(CurrentTetromino, GameBorder, tetrominos))
                    {

                        CurrentTetromino.QuickFall(GameBorder[0, 0], GameBorder[1, 1], tetrominos);
                    }

                }
                else if (key.Key == ConsoleKey.UpArrow)
                {
                    if (!CheckBottomCollision(CurrentTetromino, GameBorder, tetrominos))
                    {

                        CurrentTetromino.y = CurrentTetromino.y + 1;

                    }
                }

            }

        }

        // Function to check for collision with the top border
        static bool CheckTopCollision(int[] HighestPoints, int GameBorderY)
        {

            foreach (int point in HighestPoints)
                if (point <= GameBorderY + 1)
                {
                    return true;
                }
            return false;
        }

        // Function to check for collision with the side borders
        static bool CheckSideCollision(Tetromino ActiveBlock, int[,] GameBorder, string LeftRight, List<Tetromino> tetrominos)
        {

            if (LeftRight == "left")
            {
                foreach (int[] block in ActiveBlock.body)
                {

                    if (ActiveBlock.x + block[0] - 1 == GameBorder[0, 0])
                    {
                        return true;
                    }


                    foreach (Tetromino piece in tetrominos)
                    {

                        foreach (int[] fallenblock in piece.body)
                        {

                            if (ActiveBlock.x + block[0] - 1 == piece.x + fallenblock[0] && ActiveBlock.y + block[1] == piece.y + fallenblock[1])
                                return true;
                        }

                    }
                }
            }
            else if (LeftRight == "right")
            {
                foreach (int[] block in ActiveBlock.body)
                {
                    if (ActiveBlock.x + block[0] + 1 == GameBorder[0, 1])
                    {
                        return true;
                    }

                    foreach (Tetromino piece in tetrominos)
                    {

                        foreach (int[] fallenblock in piece.body)
                        {

                            if (ActiveBlock.x + block[0] + 1 == piece.x + fallenblock[0] && ActiveBlock.y + block[1] == piece.y + fallenblock[1])
                                return true;
                        }

                    }
                }
            }
            return false;

        }

        // Function to check for collision with the bottom border or other Tetrominos
        static bool CheckBottomCollision(Tetromino t, int[,] GameBorder, List<Tetromino> tetrominos)
        {

            foreach (int[] bottompoint in t.bottom)
            {


                if (t.y + bottompoint[1] >= GameBorder[1, 1] - 1)
                {

                    return true;
                }

                foreach (Tetromino fallenpiece in tetrominos)
                {
                    foreach (int[] fallenblock in fallenpiece.body)
                    {

                        if (t.y + bottompoint[1] + 1 == fallenpiece.y + fallenblock[1] && t.x + bottompoint[0] == fallenpiece.x + fallenblock[0])
                        {

                            return true;
                        }

                    }

                }
            }
            return false;
        }

        // Function to find the highest blocks in each column of the playing field
        static void FindHighestBlocks(List<Tetromino> tetrominos, int[,] Border, ref int[] HighestPoints)
        {
            for (int k = 0; k < HighestPoints.Length; k++)
            {
                int MinY = Border[1, 1];
                foreach (Tetromino t in tetrominos)
                {
                    foreach (int[] block in t.body)
                    {
                        if (t.x + block[0] == k + Border[0, 0] + 1 && t.y + block[1] < MinY)
                        {
                            MinY = t.y + block[1];
                        }
                    }
                }
                HighestPoints[k] = MinY;
            }

        }

    }
}