﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;

namespace ConsoleGames.Games2024
{
    class SnakeGame : Game
    {
        private int collectedApples = 0;
        private int ghostCount;
        private bool isEndlessMode = false;

        public override string Name => "Snake";
        public override string Description => "Control the snake with arrow keys";
        public override string Rules => "Try to collect apples (to become a god) without getting hit by transparent ghosts at the head.";
        public override string Credits => "Aaron Wirth";
        public override int Year => 2023;
        public override bool TheHigherTheBetter => true;
        public override int LevelMax => 4;
        public override Score HighScore { get; set; }

        private int[] playground;
        private int offsetX;
        private int offsetY;

        private string currentDirection = "right";
        private string nextDirection = "right";

        public SnakeGame()
        {
            playground = new int[] { 20, 40 };
            offsetX = (Console.WindowWidth - playground[1]) / 2;
            offsetY = (Console.WindowHeight - playground[0]) / 2;
            ghostCount = 1;
        }

        public override Score Play(int level = 1)
        {
            Score score = new Score();
            score.LevelCompleted = false;

            int scoreValue = 0;
            int[] applesRequiredForNextLevel = { 5, 10, 15, 1 };

            if (level > LevelMax) level = LevelMax;

            while (level <= LevelMax || isEndlessMode)
            {
                List<int[]> snakepos = new List<int[]> { new int[] { offsetY + 5, offsetX + 5 }, new int[] { offsetY + 5, offsetX + 6 }, new int[] { offsetY + 6, offsetX + 7 } };
                List<int[]> objects = GenerateApples(1, playground);
                List<int[]> obstacles = GenerateObstacles(level, playground, level);

                Console.Clear();
                DrawMap();
                DisplayScore(scoreValue);
                Display(snakepos, objects, obstacles);

                int initialDelay = 100 - (level * 10);
                int delay = initialDelay;

                while (true)
                {
                    string direction = UserInput();
                    if (!EvaluateTheSituation(snakepos, objects, obstacles, ref scoreValue))
                    {
                        if (isEndlessMode)
                        {

                            score.LevelCompleted = true;
                        }
                        else
                        {
                            score.LevelCompleted = false;
                            break;
                        }
                    }

                    if (objects.Count == 0)
                    {
                        List<int[]> newApples = GenerateApples(1, playground);
                        objects.AddRange(newApples);
                        collectedApples++;
                        scoreValue++;
                        DisplayScore(scoreValue);

                        if (collectedApples >= applesRequiredForNextLevel[level - 1])
                        {
                            if (level == LevelMax && !isEndlessMode)
                            {

                                obstacles = GenerateObstacles(LevelMax, playground, LevelMax);
                                isEndlessMode = true;
                            }
                            else
                            {
                                score.LevelCompleted = true;
                                DisplayLevelCompleted();
                                Thread.Sleep(2000);
                            }

                            break;
                        }
                    }

                    Movement(snakepos, objects, obstacles, ref scoreValue, direction);
                    Console.Clear();
                    DrawMap();
                    DisplayScore(scoreValue);
                    Display(snakepos, objects, obstacles);
                    Thread.Sleep(delay);
                }

                if (!score.LevelCompleted && !isEndlessMode)
                {
                    DisplayGameOver(scoreValue);
                    Console.ReadKey();
                    break;
                }
            }

            score.Points = scoreValue;
            score.Level = LevelMax;
            return score;
        }

        private List<int[]> GenerateObstacles(int count, int[] playground, int level)
        {
            List<int[]> obstacles = new List<int[]>();
            Random random = new Random();

            for (int i = 0; i < count; i++)
            {
                int newX, newY;

                do
                {
                    newX = random.Next(1, playground[1] - 1);
                    newY = random.Next(1, playground[0] - 1);
                } while (IsObstacleAtPosition(newX, newY, obstacles));

                obstacles.Add(new int[] { offsetY + newY, offsetX + newX });
            }

            return obstacles;
        }

        private List<int[]> GenerateApples(int count, int[] playground)
        {
            List<int[]> apples = new List<int[]>();
            Random random = new Random();

            for (int i = 0; i < count; i++)
            {
                int newX, newY;

                do
                {
                    newX = random.Next(1, playground[1] - 1);
                    newY = random.Next(1, playground[0] - 1);
                } while (IsAppleAtPosition(newX, newY, apples));

                apples.Add(new int[] { offsetY + newY, offsetX + newX });
            }

            return apples;
        }

        private bool IsAppleAtPosition(int x, int y, List<int[]> apples)
        {
            foreach (var applePos in apples)
            {
                if (applePos[0] == x && applePos[1] == y)
                {
                    return true;
                }
            }
            return false;
        }

        private bool IsObstacleAtPosition(int x, int y, List<int[]> obstacles)
        {
            foreach (var obstaclePos in obstacles)
            {
                if (obstaclePos[0] == x && obstaclePos[1] == y)
                {
                    return true;
                }
            }
            return false;
        }

        private void DisplayLevelCompleted()
        {
            if (!isEndlessMode)
            {
                Console.Clear();
                Console.SetCursorPosition(offsetX + 5, offsetY + playground[0] / 2);
                Console.WriteLine("LEVEL COMPLETED!");
                Console.SetCursorPosition(offsetX + 5, offsetY + playground[0] / 2 + 2);
                Console.WriteLine("Press any key to continue...");
                Console.ReadKey(true);
            }
        }

        private void DisplayScore(int score)
        {
            Console.SetCursorPosition(offsetX, offsetY + playground[0] + 2);
            Console.ForegroundColor = ConsoleColor.White;
            Console.Write($"Score: {score}");
            Console.ResetColor();
        }

        private void DisplayGameOver(int scoreValue)
        {
            Console.Clear();
            Console.SetCursorPosition(offsetX + 5, offsetY + playground[0] / 2);
            Console.WriteLine("GAME OVER!");
            Console.SetCursorPosition(offsetX + 5, offsetY + playground[0] / 2 + 2);
            Console.WriteLine($"Your Score: {scoreValue}");
            Thread.Sleep(2000);
        }

        private void Display(List<int[]> snakepos, List<int[]> objects, List<int[]> obstacles)
        {
            int currentCursorLeft = Console.CursorLeft;
            int currentCursorTop = Console.CursorTop;
            ConsoleColor currentBackgroundColor = Console.BackgroundColor;

            foreach (var pos in snakepos)
            {
                Console.SetCursorPosition(pos[1], pos[0]);
                Console.ForegroundColor = ConsoleColor.Green;
                Console.BackgroundColor = ConsoleColor.Green;
                Console.Write('*');
            }

            foreach (var apple in objects)
            {
                Console.SetCursorPosition(apple[1], apple[0]);
                Console.ForegroundColor = ConsoleColor.Red;
                Console.BackgroundColor = ConsoleColor.Red;
                Console.Write('A');
            }

            foreach (var obstacle in obstacles)
            {
                Console.SetCursorPosition(obstacle[1], obstacle[0]);
                Console.ForegroundColor = ConsoleColor.White;
                Console.BackgroundColor = ConsoleColor.White;
                Console.Write('#');
            }

            Console.SetCursorPosition(0, 0);
            Console.BackgroundColor = currentBackgroundColor;
            Console.SetCursorPosition(currentCursorLeft, currentCursorTop);
        }

        private void DrawMap()
        {
            Console.BackgroundColor = ConsoleColor.White;

            for (int i = 0; i <= playground[0] + 1; i++)
            {
                Console.SetCursorPosition(offsetX, offsetY + i);
                Console.Write(i == 0 || i == playground[0] + 1 ? "I" : " ");
                Console.SetCursorPosition(offsetX + playground[1] + 1, offsetY + i);
                Console.Write(i == 0 || i == playground[0] + 1 ? "I" : " ");
            }

            for (int i = 1; i <= playground[1]; i++)
            {
                Console.SetCursorPosition(offsetX + i, offsetY);
                Console.Write("I");
                Console.SetCursorPosition(offsetX + i, offsetY + playground[0] + 1);
                Console.Write("I");
            }

            Console.ResetColor();
        }

        private void Movement(List<int[]> snakepos, List<int[]> objects, List<int[]> obstacles, ref int score, string userInput = null)
        {
            int[] newHead = { snakepos[0][0], snakepos[0][1] };


            if (!string.IsNullOrEmpty(userInput) && !IsOppositeDirection(userInput, currentDirection))
            {
                currentDirection = userInput;
                nextDirection = userInput;
            }


            switch (currentDirection)
            {
                case "up":
                    newHead[0] -= 1;
                    break;
                case "down":
                    newHead[0] += 1;
                    break;
                case "left":
                    newHead[1] -= 1;
                    break;
                case "right":
                    newHead[1] += 1;
                    break;
                default:
                    break;
            }


            currentDirection = nextDirection;

            if (IsValidMove(newHead, snakepos, obstacles))
            {
                snakepos.Insert(0, newHead);

                if (snakepos[0][0] == -1 && snakepos[0][1] == -1)
                {
                    score++;
                    objects.Clear();
                }
                else
                {
                    snakepos.RemoveAt(snakepos.Count - 1);
                }
            }

            MoveObstacle(obstacles);
        }

        private string UserInput()
        {
            if (Console.KeyAvailable)
            {
                ConsoleKeyInfo key = Console.ReadKey(true);
                switch (key.Key)
                {
                    case ConsoleKey.UpArrow:
                        return "up";
                    case ConsoleKey.DownArrow:
                        return "down";
                    case ConsoleKey.LeftArrow:
                        return "left";
                    case ConsoleKey.RightArrow:
                        return "right";
                    default:
                        return null;
                }
            }

            return null;
        }

        private bool IsValidMove(int[] newHead, List<int[]> snakepos, List<int[]> obstacles)
        {
            foreach (var obstaclePos in obstacles.Concat(snakepos.Skip(1)))
            {
                if (newHead[0] == obstaclePos[0] && newHead[1] == obstaclePos[1])
                {
                    return false;
                }
            }

            if (newHead[0] >= offsetY + playground[0] || newHead[1] >= offsetX + playground[1] || newHead[0] <= offsetY || newHead[1] <= offsetX)
            {
                return false;
            }

            return true;
        }

        private void MoveObstacle(List<int[]> obstacles)
        {
            Random random = new Random();

            foreach (var obstaclePos in obstacles)
            {
                int direction = random.Next(4);

                switch (direction)
                {
                    case 0:
                        if (obstaclePos[0] > offsetY)
                            obstaclePos[0]--;
                        break;
                    case 1:
                        if (obstaclePos[0] < offsetY + playground[0] - 1)
                            obstaclePos[0]++;
                        break;
                    case 2:
                        if (obstaclePos[1] > offsetX)
                            obstaclePos[1]--;
                        break;
                    case 3:
                        if (obstaclePos[1] < offsetX + playground[1] - 1)
                            obstaclePos[1]++;
                        break;
                }
            }
        }

        private bool EvaluateTheSituation(List<int[]> snakepos, List<int[]> objects, List<int[]> obstacles, ref int score)
        {
            foreach (var objectPos in objects)
            {
                if (snakepos[0][0] == objectPos[0] && snakepos[0][1] == objectPos[1])
                {
                    objects.Remove(objectPos);
                    snakepos.Add(new int[] { -1, -1 });
                    score++;
                    return true;
                }
            }

            foreach (var obstaclePos in obstacles)
            {
                if (snakepos[0][0] == obstaclePos[0] && snakepos[0][1] == obstaclePos[1])
                {
                    return false;
                }
            }

            return true;
        }

        private bool IsOppositeDirection(string direction1, string direction2)
        {
            return (direction1 == "up" && direction2 == "down") ||
                   (direction1 == "down" && direction2 == "up") ||
                   (direction1 == "left" && direction2 == "right") ||
                   (direction1 == "right" && direction2 == "left");
        }
    }
}